<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Illuminate\Pagination\LengthAwarePaginator;

class DoctorController extends Controller
{
    private function decodeApiJson(string $body): ?array
    {
        $clean = trim(strip_tags($body));
        $clean = html_entity_decode($clean, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
        $clean = preg_replace('/^\xEF\xBB\xBF/', '', $clean);            // BOM
        $clean = preg_replace('/,\s*([}\]])/m', '$1', $clean);           // trailing commas

        $decoded = json_decode($clean, true);
        if (json_last_error() !== JSON_ERROR_NONE && preg_match('/\{.*\}/s', $clean, $m)) {
            $decoded = json_decode($m[0], true);
        }
        return is_array($decoded) ? $decoded : null;
    }

    public function index(Request $request)
    {
        // UI input: "specialty" is actually DEPARTMENT now
        $specialtyInput = strtoupper(trim($request->query('specialty', 'ALL')));
        $name           = $request->query('doctor');
        $keyword        = $request->query('keyword');

        $apiUrl = 'http://110.38.68.210:8080/isr/tmh/api/doctor';

        // Always fetch using specialty=MBBS (API requirement)
        $tries = [
            ['label' => 'branch=KARACHI&department=ALL&specialty=MBBS',          'q' => ['branch'=>'KARACHI','department'=>'ALL','specialty'=>'MBBS']],
            ['label' => 'branch=KARACHI&department=ALL&specialty=MBBS (repeat)', 'q' => ['branch'=>'KARACHI','department'=>'ALL','specialty'=>'MBBS']],
            ['label' => 'branch=KARACHI&specialty=MBBS (no department)',         'q' => ['branch'=>'KARACHI','specialty'=>'MBBS']],
            ['label' => 'branch=KARACHI&department=ALL (no specialty)',          'q' => ['branch'=>'KARACHI','department'=>'ALL']],
            ['label' => 'branch=KARACHI only',                                   'q' => ['branch'=>'KARACHI']],
        ];

        $raw = [];
        $used = null;
        $apiError = null;

        foreach ($tries as $t) {
            try {
                $resp = Http::timeout(10)->retry(2, 300)
                    ->withHeaders([
                        'User-Agent' => 'TMH-Laravel-Client/1.0',
                        'Accept'     => '*/*',
                    ])
                    ->get($apiUrl, $t['q']);

                if (!$resp->ok()) {
                    $apiError = 'HTTP '.$resp->status().' - '.Str::limit($resp->body(), 120);
                    continue;
                }

                $json = $this->decodeApiJson($resp->body());
                $data = $json['data'] ?? [];
                if (is_array($data) && count($data)) {
                    $raw  = array_values($data);
                    $used = $t['label'];
                    break;
                }
            } catch (\Throwable $e) {
                $apiError = $e->getMessage();
                continue;
            }
        }

        // Build specialties list from DEPARTMENT (include ALL)
        $specialties = collect($raw)->pluck('department')
            ->filter(fn($d) => filled($d) && strtoupper(trim($d)) !== 'NULL')
            ->map(fn($d) => strtoupper(trim($d)))
            ->unique()->sort()->values()
            ->prepend('ALL')->unique()->values();

        // Filter by department if a specific one is selected
        if ($specialtyInput !== 'ALL') {
            $raw = collect($raw)
                ->filter(fn ($d) => strtoupper($d['department'] ?? '') === $specialtyInput)
                ->values()
                ->all();
        }

        // Name / keyword filters and mapping
        $filtered = collect($raw)->filter(function ($d) use ($name, $keyword) {
            if ($name && !Str::of($d['name'] ?? '')->lower()->contains(strtolower($name))) return false;
            if ($keyword) {
                $hay = Str::of(
                    ($d['name'] ?? '').' '.($d['department'] ?? '').' '.($d['specialty'] ?? '').' '.($d['education'] ?? '')
                )->lower();
                if (!$hay->contains(strtolower($keyword))) return false;
            }
            return true;
        })->map(function ($d) {
            $gender = strtolower($d['gender'] ?? '');
            $imgUrl = $d['image_url'] ?? 'not uploaded';
            if ($imgUrl === 'not uploaded' || blank($imgUrl)) {
                $imgUrl = $gender === 'male'
                    ? asset('assets/images/resource/find-doctors/placeholder-male.jpg')
                    : asset('assets/images/resource/find-doctors/placeholder-female.jpg');
            }
            $schedule = [];
            if (!empty($d['schedule']) && is_array($d['schedule'])) {
                foreach ($d['schedule'] as $day => $slot) {
                    $schedule[] = ['day'=>$day, 'start'=>$slot['start'] ?? null, 'end'=>$slot['end'] ?? null];
                }
            }
            return [
                'id'         => $d['id'] ?? null,
                'name'       => $d['name'] ?? '',
                'gender'     => $d['gender'] ?? '',
                'department' => $d['department'] ?? '',
                'specialty'  => $d['specialty'] ?? '',
                'contact'    => $d['contact'] ?? '',
                'education'  => $d['education'] ?? '',
                'branch'     => $d['branch'] ?? '',
                'image'      => $imgUrl,
                'schedule'   => $schedule,
            ];
        })->values();

        // ---- PAGINATION (12 per page) ----
        $perPage = 12;
        $page    = LengthAwarePaginator::resolveCurrentPage();
        $total   = $filtered->count();

        $currentPageItems = $filtered->forPage($page, $perPage)->values();

        $doctorsPaginator = new LengthAwarePaginator(
            $currentPageItems,
            $total,
            $perPage,
            $page,
            [
                'path'     => url()->current(),
                'pageName' => 'page'
            ]
        );

        return view('pages.find-a-doctor', [
            'doctors'     => $doctorsPaginator,  // <-- paginator
            'specialties' => $specialties,       // departments
            'specialty'   => $specialtyInput,    // selected department
            'name'        => $name,
            'keyword'     => $keyword,
            'apiError'    => $apiError,
            'apiCount'    => is_countable($raw) ? count($raw) : 0,
            'apiVariant'  => $used,
        ]);
    }
}
